import { AgoraRTCError as AgoraRTCError_2 } from '@agora-js/shared';
import { AgoraRTCErrorCode } from '@agora-js/shared';
import { EventEmitter } from '@agora-js/shared';
import { NETWORK_STATE } from '@agora-js/shared';
import type { SDK_MODE } from '@agora-js/shared';
import type { UID } from '@agora-js/shared';

export declare interface AgoraApiExecutor<T> {
    onSuccess: (result: T) => void;
    onError: (err: AgoraRTCError_2) => void;
}

export declare enum AgoraEventUploadId {
    WORKER_EVENT = 156,
    AP_WORKER_EVENT = 160
}

/**
 * 控制 SDK 日志输出的对象，可以通过全局的 `Logger` 对象获取
 */
export declare class AgoraLogger {
    private proxyServerURL?;
    private logLevel;
    private uploadState;
    /** 等待上传的 log 数据 */
    private uploadLogWaitingList;
    /** 正在上传的 log 数据，为空表示没有正在上传的 log 数据 */
    private uploadLogUploadingList;
    /** 连续上传失败的次数 */
    private uploadErrorCount;
    /**
     * 累加的 log id，用于 log 上报
     */
    private currentLogID;
    /**
     * for test
     */
    private url?;
    /** @internal */
    constructor();
    /** @internal */
    debug(...args: any): void;
    /** @internal */
    info(...args: any): void;
    /** @internal */
    warning(...args: any): void;
    /** @internal */
    warn(...args: any): void;
    /** @internal */
    error(...args: any): void;
    /** @internal */
    upload(...args: any): void;
    /**
     * 设置 SDK 的日志输出级别
     * @param level - SDK 日志级别依次为 NONE(4)，ERROR(3)，WARNING(2)，INFO(1)，DEBUG(0)。选择一个级别，
     * 你就可以看到在该级别及该级别以上所有级别的日志信息。
     *
     * 例如，如果你输入代码 AgoraRTC.Logger.setLogLevel(1);，就可以看到 INFO，ERROR 和 WARNING 级别的日志信息。
     */
    setLogLevel(level: number): void;
    /**
     * 调用本方法开启日志上传。开启后 SDK 的日志会上传到声网的服务器。
     *
     * 日志上传功能默认为关闭状态，如果你需要开启此功能，请确保在所有方法之前调用本方法。
     *
     * **Note:** 如果没有成功加入频道，则服务器上无法查看日志信息。
     */
    enableLogUpload(): void;
    /**
     * 该方法用于关闭日志上传。
     *
     * 日志上传默认是关闭状态，如果你调用了开启日志上传（[enableLogUpload]{@link IAgoraLogger.enableLogUpload})，可以通过本方法停止上传日志。
     */
    disableLogUpload(): void;
    /** @internal */
    setProxyServer(proxyServerURL?: string): void;
    /** @internal */
    prefix(prefix: string): EnhancedAgoraLogger;
    private log;
    private uploadLog;
    private appendLogToWaitingList;
    startUpload(): void;
    private uploadLogs;
    private uploadLogInterval;
    extLog: (level: number, argsWithStyle: string[]) => void;
}

/**
 * @ignore
 */
/** @en
 * @ignore
 */
export declare class AgoraRTCError extends AgoraRTCError_2 {
    readonly name: string;
    constructor(code: AgoraRTCErrorCode, message?: string, data?: any);
    print(level?: "error" | "warning"): AgoraRTCError;
    throw(): never;
}

export { AgoraRTCErrorCode }

export declare enum AgoraRTCEvent {
    PUBLISH = "publish",
    SUBSCRIBE = "subscribe",
    WS_COMPRESSOR_INIT = "ws_compressor_init",
    SESSION_INIT = "session_init",
    JOIN_CHOOSE_SERVER = "join_choose_server",
    REQ_USER_ACCOUNT = "req_user_account",
    JOIN_GATEWAY = "join_gateway",
    REJOIN_GATEWAY = "rejoin_gateway",
    STREAM_SWITCH = "stream_switch",
    REQUEST_PROXY_WORKER_MANAGER = "request_proxy_worker_manager",
    REQUEST_PROXY_APPCENTER = "request_proxy_appcenter",
    FIRST_VIDEO_RECEIVED = "first_video_received",
    FIRST_AUDIO_RECEIVED = "first_audio_received",
    FIRST_VIDEO_DECODE = "first_video_decode",
    FIRST_AUDIO_DECODE = "first_audio_decode",
    ON_ADD_AUDIO_STREAM = "on_add_audio_stream",
    ON_ADD_VIDEO_STREAM = "on_add_video_stream",
    ON_UPDATE_STREAM = "on_update_stream",
    ON_REMOVE_STREAM = "on_remove_stream",
    USER_ANALYTICS = "req_user_analytics",
    PC_STATS = "pc_stats",
    UPDATE_REMOTE_RTPCAPABILITIES = "update_remote_rtpCapabilities"
}

export declare type AgoraRTCEventFirstRemoteFrame = AgoraRTCEvent.FIRST_AUDIO_DECODE | AgoraRTCEvent.FIRST_AUDIO_RECEIVED | AgoraRTCEvent.FIRST_VIDEO_DECODE | AgoraRTCEvent.FIRST_VIDEO_RECEIVED;

export declare type AgoraRTCEventOnGatewayStream = AgoraRTCEvent.ON_ADD_AUDIO_STREAM | AgoraRTCEvent.ON_ADD_VIDEO_STREAM | AgoraRTCEvent.ON_UPDATE_STREAM | AgoraRTCEvent.ON_REMOVE_STREAM;

export declare class AgoraRTCEventReport {
    /**
     * 以 sessionID 为 key 存储每次上报都需要带上的基本信息
     */
    private baseInfoMap;
    private proxyServer?;
    /**
     * 内部循环，用于定时上报事件
     */
    private eventUploadTimer;
    /**
     * 内部循环，用于定时检测是否有加入了频道的 client 使得无 sessionId 的 ApiInvoke 事件可以被上报。
     */
    private setSessionIdTimer;
    /**
     * for test
     */
    private url?;
    private sids;
    /**
     * for test
     */
    private backupUrl?;
    private _appId?;
    private _aid;
    /**
     * 等待上报的事件信息
     */
    /** 高优先级的上报信息，立刻上报不会合并 */
    private keyEventUploadPendingItems;
    /** 低优先级的上报信息，会根据时间和事件数合并上报 */
    private normalEventUploadPendingItems;
    /**
     * NOTE:
     * 高优先级的上报逻辑：
     *  - 触发后立刻加入 keyEvents 队列
     *  - 加入队列后立刻将队列中全部数据上报，清空队列
     * > 如果此时队列长度 > 20, 只上报最先入队的 20 个，其余的留在队列中
     *  - 如果上报失败，将这些数据加回 keyEvents 队列，但是不会触发重试
     *
     * 低优先级上报逻辑：
     *  - 触发后立刻加入 normalEvents 队列
     * > 如果此时队列长度 > 100, drop 最早入队的数据
     *  - 如果 normalEvents 长度 > 10, 将队列中的数据全部上报，清空队列
     * > 如果此时队列长度 > 20, 只上报最先入队的 20 个，其余的留在队列中
     *  - 如果上报失败，将这些数据加回 normalEvents 队列，但是不会触发重试
     *
     * 同时，reports 还会启动一个 3s 执行一次的定时器，定时器的逻辑：
     *  - 检查 keyEvents 队列，如果发现有值，立刻全部上报并清空队列
     *  - 检查 normalEvents 队列，如果发现有值并且距离上一次上报 normalEvents 已经过去了 5s，将队列中的数据全部上报，清空队列
     * > 如果此时队列长度 > 20, 只上报最先入队的 20 个，其余的留在队列中
     *
     */
    /**
     * 如果调用 api 时还没有拿到 session id，就先缓存在这里
     */
    private apiInvokeUploadPendingItems;
    private apiInvokeCount;
    private apiInvokeLoggedCount;
    /** 防止上报数据的 lts 字段重复 */
    private ltsList;
    /** 上一次上报低优先级事件的时间 */
    private lastSendNormalEventTime;
    private customReportCounterTimer?;
    private customReportCount;
    constructor();
    getBaseInfoBySessionId(sessionId: string): {
        info: EventReportBase;
        startTime: number;
        lastJoinSuccessTime?: number;
    };
    setAppId(appid: string): void;
    reportApiInvoke<T>(sessionId: string | null, params: EventReportReportAPIInvokeParams<T>, throttleTime?: number): AgoraApiExecutor<T>;
    sessionInit(sessionId: string, params: EventReportInitParams): void;
    joinChooseServer(sessionId: string, params: EventReportChooseServerParams): void;
    reqUserAccount(sessionId: string, params: EventReportReqUserAccountParams): void;
    joinGateway(sessionId: string, params: EventReportJoinGatewayParams): void;
    joinChannelTimeout(sessionId: string, timeout: number): void;
    publish(sessionId: string, params: EventReportPublishParams): void;
    subscribe(sessionId: string, params: EventReportSubscribeParams, isMassSubscribe?: boolean): void;
    wsCompressorInit(params: EventReportWsCompressorInitParams): void;
    firstRemoteVideoDecode(sessionId: string, eventType: AgoraRTCEventFirstRemoteFrame, uploadType: AgoraRTCEventUploadFirstRemoteFrame, params: EventFirstRemoteVideoFrameParams): void;
    firstRemoteFrame(sessionId: string, eventType: AgoraRTCEventFirstRemoteFrame, uploadType: AgoraRTCEventUploadFirstRemoteFrame, params: EventFirstRemoteFrameParams): void;
    pcStats(sessionId: string, params: EventPCStatsParams): void;
    updateRemoteRTPCapabilities(sessionId: string | null, params: EventUpdateRemoteRTPCapabilitiesParams): void;
    onGatewayStream(sessionId: string, eventType: AgoraRTCEventOnGatewayStream, uploadType: AgoraRTCEventUploadOnGatewayStream, params: EventOnGatewayStreamParams): void;
    streamSwitch(sessionId: string, params: EventReportStreamSwitchParams): void;
    requestProxyAppCenter(sessionId: string, params: EventReportRequestProxyAppCenterParams): void;
    requestProxyWorkerManager(sessionId: string, params: EventReportRequestProxyWorkerManagerParams): void;
    setProxyServer(url?: string): void;
    peerPublishStatus(sessionId: string, params: EventPeerPublishStatusParams): void;
    workerEvent(sessionId: string, params: IWorkerEvent): void;
    apworkerEvent(sessionId: string, params: IAPWorkerEvent): void;
    joinWebProxyAP(sessionId: string, params: IJoinWebProxyAP): void;
    WebSocketQuit(sessionId: string, params: IWebSocketQuit): void;
    sendCustomReportMessage(sessionId: string, params: EventCustomReportParams[]): Promise<void>;
    private sendApiInvoke;
    addSid(sid: string): void;
    removeSid(sid: string): void;
    private appendSessionId;
    send(item: AgoraRTCEventUploadItem, isKeyEvent: boolean): void;
    private doSend;
    private sendItems;
    private postDataToStatsCollector2;
    private postDataToStatsCollector;
    private createBaseInfo;
    reportResourceTiming(url: string, sessionId: string): void;
    extApiInvoke: (msgs: ExtensionApiInvokeMessage[]) => Promise<void>;
}

export declare type AgoraRTCEventUploadFirstRemoteFrame = AgoraRTCEventUploadType.FIRST_AUDIO_DECODE | AgoraRTCEventUploadType.FIRST_AUDIO_RECEIVED | AgoraRTCEventUploadType.FIRST_VIDEO_DECODE | AgoraRTCEventUploadType.FIRST_VIDEO_RECEIVED;

export declare interface AgoraRTCEventUploadItem {
    type: AgoraRTCEventUploadType;
    data: EventReportBase | newReportMessages;
    isPb?: boolean;
    id?: number;
}

export declare interface AgoraRTCEventUploadJsonItem {
    type: AgoraRTCEventUploadType;
    data: EventReportBase;
    isPb?: boolean;
}

export declare type AgoraRTCEventUploadOnGatewayStream = AgoraRTCEventUploadType.ON_ADD_AUDIO_STREAM | AgoraRTCEventUploadType.ON_ADD_VIDEO_STREAM | AgoraRTCEventUploadType.ON_UPDATE_STREAM | AgoraRTCEventUploadType.ON_REMOVE_STREAM;

export declare enum AgoraRTCEventUploadType {
    SESSION = "io.agora.pb.Wrtc.Session",
    JOIN_CHOOSE_SERVER = "io.agora.pb.Wrtc.JoinChooseServer",
    REQ_USER_ACCOUNT = "io.agora.pb.Wrtc.ReqUserAccount",
    JOIN_GATEWAY = "io.agora.pb.Wrtc.JoinGateway",
    RE_JOIN_GATEWAY = "io.agora.pb.Wrtc.ReJoinGateway",
    PUBLISH = "io.agora.pb.Wrtc.Publish",
    SUBSCRIBE = "io.agora.pb.Wrtc.Subscribe",
    WS_COMPRESSOR_INIT = "io.agora.pb.Wrtc.WsCompressorInit",
    STREAM_SWITCH = "io.agora.pb.Wrtc.StreamSwitch",
    AUDIO_SENDING_STOPPED = "io.agora.pb.Wrtc.AudioSendingStopped",
    VIDEO_SENDING_STOPPED = "io.agora.pb.Wrtc.VideoSendingStopped",
    REQUEST_PROXY_APPCENTER = "io.agora.pb.Wrtc.RequestProxyAppCenter",
    REQUEST_PROXY_WORKER_MANAGER = "io.agora.pb.Wrtc.RequestProxyWorkerManager",
    API_INVOKE = "io.agora.pb.Wrtc.ApiInvoke",
    FIRST_VIDEO_RECEIVED = "io.agora.pb.Wrtc.FirstVideoReceived",
    FIRST_AUDIO_RECEIVED = "io.agora.pb.Wrtc.FirstAudioReceived",
    FIRST_VIDEO_DECODE = "io.agora.pb.Wrtc.FirstVideoDecode",
    FIRST_AUDIO_DECODE = "io.agora.pb.Wrtc.FirstAudioDecode",
    ON_ADD_AUDIO_STREAM = "io.agora.pb.Wrtc.OnAddAudioStream",
    ON_ADD_VIDEO_STREAM = "io.agora.pb.Wrtc.OnAddVideoStream",
    ON_UPDATE_STREAM = "io.agora.pb.Wrtc.OnUpdateStream",
    ON_REMOVE_STREAM = "io.agora.pb.Wrtc.OnRemoveStream",
    JOIN_CHANNEL_TIMEOUT = "io.agora.pb.Wrtc.JoinChannelTimeout",
    PEER_PUBLISH_STATUS = "io.agora.pb.Wrtc.PeerPublishStatus",
    WORKER_EVENT = "io.agora.pb.Wrtc.WorkerEvent",
    AP_WORKER_EVENT = "io.agora.pb.Wrtc.APWorkerEvent",
    JOIN_WEB_PROXY_AP = "io.agora.pb.Wrtc.JoinWebProxyAP",
    WEBSOCKET_QUIT = "io.agora.pb.Wrtc.WebSocketQuit",
    USER_ANALYTICS = "io.agora.pb.Wrtc.UserAnalytics",
    AUTOPLAY_FAILED = "io.agora.pb.Wrtc.AutoplayFailed",
    PC_STATS = "io.agora.pb.Wrtc.PCStats",
    UPDATE_REMOTE_RTPCAPABILITIES = "io.agora.pb.Wrtc.UpdateRemoteRTPCapabilities"
}

export declare function apiInvoke(options?: {
    report?: AgoraRTCEventReport;
    reportResult?: boolean;
    className?: string;
    argsMap?: (self: any, ...args: any[]) => any[];
    throttleTime?: number;
}): ApiInvokeDecorator;

export declare type ApiInvokeDecorator = <T>(target: T, propertyKey: any, descriptor: TypedPropertyDescriptor<any>) => TypedPropertyDescriptor<any> | undefined;

export declare interface ApiInvokeMessage {
    tag?: string;
    sid: null | string;
    name: string;
    invokeId: number;
    result?: any;
    states?: any;
    options?: any;
    error?: string | AgoraRTCError_2;
    success: boolean;
    apiInvokeTime: number;
    lts?: number;
}

export declare type APIUploadData = [
...BaseUploadData,
[
string,
string | undefined,
string | undefined
]
];

export declare type BaseUploadData = [
number,
string,
LOG_TYPE,
string,
string | number | undefined
];

declare class EnhancedAgoraLogger {
    private logger;
    private prefixLists;
    constructor(logger: AgoraLogger);
    debug(...args: any): void;
    info(...args: any): void;
    warning(...args: any): void;
    error(...args: any): void;
    prefix(prefix: string): EnhancedAgoraLogger;
    popPrefix(): EnhancedAgoraLogger;
}

export declare const EVENT_BASE_TEMPLATE: EventReportBase;

/**
 * 自定义事件上报的参数，用于 [AgoraRTCClient.sendCustomReportMessage]{@link IAgoraRTCClient.sendCustomReportMessage}。
 */
/** @en
 * Parameters for reporting customized messages. Used when calling [AgoraRTCClient.sendCustomReportMessage]{@link IAgoraRTCClient.sendCustomReportMessage}.
 */
export declare interface EventCustomReportParams {
    /**
     * 该条上报的信息的 ID。
     */
    /** @en
     * The ID of the message.
     */
    reportId: string;
    /** @en
     * The category of the message.
     */
    category: string;
    /**
     * 该条上报的信息的事件名。
     */
    /** @en
     * The event name of the message.
     */
    event: string;
    /**
     * 该条上报的信息的标签。
     */
    /** @en
     * The label of the message.
     */
    label: string;
    /**
     * 该条上报的信息携带的值。
     */
    /** @en
     * The value of the message.
     */
    value: number;
}

export declare interface EventFirstRemoteFrameMessage extends EventReportBase {
    eventType: AgoraRTCEventFirstRemoteFrame;
    peer: number;
    elapse: number;
    subscribeElapse: number;
    subscribeRequestid: number;
    p2pid: number;
    videowidth?: number;
    videoheight?: number;
}

export declare interface EventFirstRemoteFrameParams {
    peer: number;
    subscribeElapse: number;
    subscribeRequestid: number;
    p2pid: number;
    videowidth?: number;
    videoheight?: number;
}

export declare interface EventFirstRemoteVideoDecodeMessage extends EventReportBase {
    eventType: AgoraRTCEventFirstRemoteFrame;
    peer: number;
    elapse: number;
    subscribeElapse: number;
    subscribeRequestid: number;
    p2pid: number;
    videowidth?: number;
    videoheight?: number;
    apStart: number;
    apEnd: number;
    joinGwStart: number;
    joinGwEnd: number;
    pcStart: number;
    pcEnd: number;
    videoAddNotify: number;
    subscriberStart: number;
    subscriberEnd: number;
    firstDecodeFrame: number;
    state: number;
}

export declare interface EventFirstRemoteVideoFrameParams {
    peer: number;
    subscribeElapse: number;
    subscribeRequestid: number;
    p2pid: number;
    videowidth?: number;
    videoheight?: number;
    apStart: number;
    apEnd: number;
    joinGwStart: number;
    joinGwEnd: number;
    pcStart: number;
    pcEnd: number;
    videoAddNotify: number;
    subscriberStart: number;
    subscriberEnd: number;
    state: number;
    firstFrame?: number;
}

export declare interface EventJoinChannelTimeoutMessage extends EventReportBase {
    timeout: number;
    elapse: number;
}

export declare enum EventNameToID {
    SESSION = 26,
    JOIN_CHOOSE_SERVER = 27,
    REQ_USER_ACCOUNT = 196,
    JOIN_GATEWAY = 28,
    PUBLISH = 30,
    SUBSCRIBE = 29,
    WS_COMPRESSOR_INIT = 9430,
    STREAM_SWITCH = 32,
    AUDIO_SENDING_STOPPED = 33,
    VIDEO_SENDING_STOPPED = 34,
    REQUEST_PROXY_APPCENTER = 35,
    REQUEST_PROXY_WORKER_MANAGER = 36,
    API_INVOKE = 41,
    FIRST_VIDEO_RECEIVED = 348,
    FIRST_AUDIO_RECEIVED = 349,
    FIRST_VIDEO_DECODE = 350,
    FIRST_AUDIO_DECODE = 351,
    ON_ADD_AUDIO_STREAM = 352,
    ON_ADD_VIDEO_STREAM = 353,
    ON_UPDATE_STREAM = 356,
    ON_REMOVE_STREAM = 355,
    JOIN_CHANNEL_TIMEOUT = 407,
    PEER_PUBLISH_STATUS = 408,
    WORKER_EVENT = 156,
    AP_WORKER_EVENT = 160,
    JOIN_WEB_PROXY_AP = 700,
    WEBSOCKET_QUIT = 671,
    USER_ANALYTICS = 10000,
    AUTOPLAY_FAILED = 9178
}

export declare interface EventOnGatewayStreamMessage extends EventReportBase {
    eventType: AgoraRTCEventOnGatewayStream;
    peer: number;
    audio?: boolean;
    video?: boolean;
}

export declare interface EventOnGatewayStreamParams {
    peer: number;
    ssrc?: number;
    audio?: boolean;
    video?: boolean;
}

export declare interface EventPCStatsParams {
    startTime: number;
    eventElapse: number;
    iceconnectionsate: RTCIceConnectionState;
    dtlsstate: RTCDtlsTransportState | null;
    connectionstate: RTCPeerConnectionState;
    intSucc: number;
    error: null | string;
    selectedLocalCandidateProtocol: string;
    selectedLocalCandidateType: string;
    selectedLocalCandidateAddress: string;
    selectedRemoteCandidateProtocol: string;
    selectedRemoteCandidateType: string;
    selectedRemoteCandidateAddress: string;
    restartCnt: number;
    preallocation?: boolean;
}

export declare interface EventPeerPublishStatusMessage extends EventReportBase {
    joinChannelSuccessElapse: number;
    subscribeElapse: number;
    peerPublishDuration: number;
    peer: number;
    audiotag: number;
    videotag: number;
    peerPublishDurationVideo: number;
    peerPublishDurationAudio: number;
}

export declare interface EventPeerPublishStatusParams {
    subscribeElapse: number;
    peer: number;
    audioPublishDuration: number;
    videoPublishDuration: number;
}

export declare interface EventReportApiInvokeMessage extends EventReportBase {
    invokeId: number;
    execElapse: number;
    options?: string;
    execStates?: string;
    errorCode?: string;
    errorMsg?: string;
    execResult?: string;
    apiName: string;
}

export declare interface EventReportBase {
    sid: string;
    lts: number;
    success: boolean | null;
    cname: string | null;
    uid: number | null;
    peer?: number | null;
    cid: number | null;
    elapse: number | null;
    extend?: string | null;
    vid?: number | string;
    eventType?: string;
    apiName?: string;
}

export declare interface EventReportChooseServerParams {
    role?: "host" | "audience";
    lts: number;
    csAddr: string;
    ec: null | string;
    succ: boolean;
    serverList: null | string[];
    uid?: string;
    cid?: string;
    csIp?: string;
    unilbsServerIds?: string;
    extend?: string;
    opid: number;
    isHttp3?: 1 | -1;
    elapse?: number;
    corssRegionTagReq?: string;
    corssRegionTagRes?: string;
}

export declare interface EventReportInitParams {
    lts: number;
    buildFormat: number;
    /** channel name */
    cname: string;
    appid: string;
    mode: SDK_MODE;
    extend?: {
        rejoin?: boolean;
        recover?: boolean;
        willUploadConsoleLog?: boolean;
    };
    stringUid?: string;
    channelProfile?: number;
    channelMode?: number;
    isABTestSuccess?: number;
    lsid?: string;
    fsid?: string;
    clientType?: 20;
    clientRole?: number;
    info?: string;
}

export declare interface EventReportJoinChooseServerMessage extends EventReportBase {
    role?: "host" | "audience";
    eventType: AgoraRTCEvent.JOIN_CHOOSE_SERVER;
    eventElapse: number;
    chooseServerAddr: string;
    errorCode: string | null;
    chooseServerAddrList: string;
    chooseServerIp: string;
    unilbsServerIds?: string;
    extend?: string;
    opid: number;
    isHttp3?: 1 | -1;
    corssRegionTagReq?: string;
    corssRegionTagRes?: string;
}

export declare interface EventReportJoinGatewayMessage extends EventReportBase {
    eventType: AgoraRTCEvent.JOIN_GATEWAY;
    gatewayAddr: null | string;
    errorCode: null | string;
    errorMsg: null | string;
    eventElapse: number;
    firstSuccess: boolean;
    signalChannel: string;
    preload: number;
}

export declare interface EventReportJoinGatewayParams {
    vid?: string;
    lts: number;
    succ: boolean;
    ec: null | string;
    addr: null | string;
    uid: number;
    cid: number;
    firstSuccess: boolean;
    avoidJoinStartTime: number;
    isProxy: boolean;
    signalChannel: string;
    errorMsg?: string;
    preload?: boolean;
}

export declare interface EventReportMediaSendingStoppedParams {
    reason: string;
    succ: boolean;
}

export declare interface EventReportPublishMessage extends EventReportBase {
    eventType: AgoraRTCEvent.PUBLISH;
    eventElapse: number;
    errorCode: null | string;
    videoName?: string;
    audioName?: string;
    screenName?: string;
    audio: boolean;
    video: boolean;
    screenshare: boolean;
    publishRequestid: number;
    p2pid: number;
}

export declare interface EventReportPublishParams {
    eventElapse: number;
    succ: boolean;
    audio: boolean;
    video: boolean;
    screenshare: boolean;
    p2pid: number;
    publishRequestid: number;
    audioName?: string;
    videoName?: string;
    screenName?: string;
    ec: null | string;
    extend?: string;
}

export declare interface EventReportReJoinGatewayMessage extends NotContainSuccessPropertyEventReportBase {
    eventType: AgoraRTCEvent.REJOIN_GATEWAY;
    gatewayAddr: null | string;
    errorCode: null | string;
    eventElapse: number;
    isSuccess: 0 | 1;
    gatewayIp: string;
    port: string;
    isProxy: 0 | 1;
}

export declare interface EventReportReportAPIInvokeParams<T> {
    tag?: string;
    id?: string | number;
    name: string;
    options: any;
    states?: any;
    timeout?: number;
    reportResult?: boolean;
}

export declare interface EventReportRequestProxyAppCenterMessage extends EventReportBase {
    eventType: AgoraRTCEvent.REQUEST_PROXY_APPCENTER;
    eventElapse: number;
    APAddr: string;
    workerManagerList: null | string;
    response: string;
    errorCode: null | string;
}

export declare interface EventReportRequestProxyAppCenterParams {
    lts: number;
    succ: boolean;
    APAddr: string;
    workerManagerList: null | string;
    ec: null | string;
    response: string;
}

export declare interface EventReportRequestProxyWorkerManagerMessage extends EventReportBase {
    eventType: AgoraRTCEvent.REQUEST_PROXY_WORKER_MANAGER;
    eventElapse: number;
    workerManagerAddr: string;
    response: string;
    errorCode: null | string;
}

export declare interface EventReportRequestProxyWorkerManagerParams {
    lts: number;
    succ: boolean;
    workerManagerAddr: string;
    ec: null | string;
    response: string;
}

export declare interface EventReportReqUserAccountMessage extends EventReportBase {
    eventType: AgoraRTCEvent.REQ_USER_ACCOUNT;
    serverAddress: string;
    stringUid: string;
    errorCode: null | string;
    eventElapse: number;
}

export declare interface EventReportReqUserAccountParams {
    lts: number;
    elapse?: number;
    success: boolean;
    serverAddr: string;
    stringUid: string;
    uid: null | number;
    errorCode: null | string;
    extend: any;
}

export declare interface EventReportSessionInitMessage extends EventReportBase {
    eventType: AgoraRTCEvent.SESSION_INIT;
    appid: string;
    browser: string;
    buildFormat: number;
    build: string;
    mode: SDK_MODE;
    process: string;
    version: string;
    appType: number;
    stringUid?: string;
    channelProfile?: number;
    channelMode?: number;
    isABTestSuccess?: number;
    lsid?: string;
    fsid?: string;
    clientType?: number;
    clientRole?: number;
    serviceId?: string;
    extensionID?: string;
    info?: string;
}

export declare interface EventReportStreamSwitchMessage extends EventReportBase {
    eventType: AgoraRTCEvent.STREAM_SWITCH;
    isDual: boolean;
}

export declare interface EventReportStreamSwitchParams {
    lts: number;
    isdual: boolean;
    succ: boolean;
}

export declare interface EventReportSubscribeMessage extends EventReportBase {
    eventType: AgoraRTCEvent.SUBSCRIBE;
    eventElapse: number;
    errorCode: null | string;
    peerSuid?: string;
    video: boolean;
    audio: boolean;
    subscribeRequestid: number;
    p2pid: number;
    preSsrc?: number;
}

export declare interface EventReportSubscribeParams {
    succ: boolean;
    video: boolean;
    audio: boolean;
    peerid: UID;
    ec: null | string;
    subscribeRequestid: number;
    p2pid: number;
    eventElapse: number;
    preSsrc?: boolean;
}

export declare interface EventReportWsCompressorInitMessage extends EventReportBase {
    eventType: AgoraRTCEvent.WS_COMPRESSOR_INIT;
    eventElapse: number;
    status: 1 | 2;
}

export declare interface EventReportWsCompressorInitParams {
    status: boolean;
    ec: null | string;
    eventElapse: number;
}

export declare type EventUpdateRemoteRTPCapabilitiesMessage = EventUpdateRemoteRTPCapabilitiesParams & EventReportBase;

export declare interface EventUpdateRemoteRTPCapabilitiesParams {
    /**
     * 实际发流的type
     * types 为空时，表示此时没有发流
     */
    trackTypes: string;
    /**
     * 本地媒体能力
     */
    localCodecs: string;
    /**
     * 对端媒体能力
     */
    remoteCodecs: string;
}

export declare type EventUploadData = [...BaseUploadData, [string, string | undefined]];

export declare interface ExtensionApiInvokeMessage {
    name: string;
    result?: any;
    states?: any;
    options?: any;
    error?: string;
    success: boolean;
    apiInvokeTime: number;
    lts?: number;
}

export declare interface IAPWorkerEvent {
    /** APWorkerEvent sid */
    sid?: string | null;
    /** APWorkerEvent cname */
    cname?: string | null;
    /** APWorkerEvent cid */
    cid?: number | Long_2 | null;
    /** APWorkerEvent lts */
    lts?: number | Long_2 | null;
    /** APWorkerEvent ip */
    ip?: string | null;
    /** APWorkerEvent uid */
    uid?: number | Long_2 | null;
    /** APWorkerEvent success */
    success?: boolean | null;
    /** APWorkerEvent elapse */
    elapse?: number | Long_2 | null;
    /** APWorkerEvent peer */
    peer?: number | Long_2 | null;
    /** APWorkerEvent ec */
    ec?: number | null;
    /** APWorkerEvent sc */
    sc?: number | null;
    /** APWorkerEvent serverIp */
    serverIp?: string | null;
    /** APWorkerEvent firstSuccess */
    firstSuccess?: boolean | null;
    /** APWorkerEvent responseTime */
    responseTime?: number | null;
    /** APWorkerEvent serviceName */
    serviceName?: string | null;
    /** APWorkerEvent responseDetail */
    responseDetail?: string | null;
}

export declare interface IJoinWebProxyAP {
    /** JoinWebProxyAP lts */
    lts?: number | Long_2 | null;
    /** JoinWebProxyAP sid */
    sid?: string | null;
    /** JoinWebProxyAP cname */
    cname?: string | null;
    /** JoinWebProxyAP cid */
    cid?: number | Long_2 | null;
    /** JoinWebProxyAP ip */
    ip?: string | null;
    /** JoinWebProxyAP uid */
    uid?: number | Long_2 | null;
    /** JoinWebProxyAP sucess */
    sucess?: number | null;
    /** JoinWebProxyAP elapse */
    elapse?: number | Long_2 | null;
    /** JoinWebProxyAP eventType */
    eventType?: string | null;
    /** JoinWebProxyAP apServerAddr */
    apServerAddr?: string | null;
    /** JoinWebProxyAP turnServerAddrList */
    turnServerAddrList?: string | null;
    /** JoinWebProxyAP errorCode */
    errorCode?: string | null;
    unilbsServerIds?: string;
    extend?: string;
}

export declare interface InternalAPILog extends LogData {
    type: LOG_TYPE.INTERNAL_API;
    api: string;
    args?: string;
    result?: string;
}

export declare interface InternalEventLog extends LogData {
    type: LOG_TYPE.INTERNAL_EVENT;
    event: string;
    msg?: string;
}

export declare function isEventCustomReportParams(params: EventCustomReportParams): params is EventCustomReportParams;

export declare interface IWebSocketQuit {
    /** WebSocketQuit lts */
    lts?: number | Long_2 | null;
    /** WebSocketQuit sid */
    sid?: string | null;
    /** WebSocketQuit cname */
    cname?: string | null;
    /** WebSocketQuit cid */
    cid?: number | Long_2 | null;
    /** WebSocketQuit uid */
    uid?: number | Long_2 | null;
    /** WebSocketQuit elapse */
    elapse?: number | Long_2 | null;
    /** WebSocketQuit succ */
    succ?: number | null;
    /** WebSocketQuit errorCode */
    errorCode?: string | null;
    /** WebSocketQuit ip */
    ip?: string | null;
}

export declare interface IWorkerEvent {
    /** WorkerEvent sid */
    sid?: string | null;
    /** WorkerEvent cname */
    cname?: string | null;
    /** WorkerEvent cid */
    cid?: number | Long_2 | null;
    /** WorkerEvent lts */
    lts?: number | Long_2 | null;
    /** WorkerEvent ip */
    ip?: string | null;
    /** WorkerEvent uid */
    uid?: number | Long_2 | null;
    /** WorkerEvent success */
    success?: boolean | null;
    /** WorkerEvent elapse */
    elapse?: number | Long_2 | null;
    /** WorkerEvent peer */
    peer?: number | Long_2 | null;
    /** WorkerEvent command */
    command?: string | null;
    /** WorkerEvent actionType */
    actionType?: string | null;
    /** WorkerEvent url */
    url?: string | null;
    /** WorkerEvent payload */
    payload?: string | null;
    /** WorkerEvent serverCode */
    serverCode?: number | null;
    /** WorkerEvent code */
    code?: number | null;
    /** WorkerEvent traceId */
    traceId?: string | null;
    /** WorkerEvent workerType */
    workerType?: number | null;
    /** WorkerEvent responseTime */
    responseTime?: number | null;
    /** WorkerEvent requestId */
    requestId?: number | Long_2 | null;
    /** WorkerEvent packIndex */
    packIndex?: number | null;
    /** WorkerEvent requestByUser */
    requestByUser?: boolean | null;
    /** WorkerEvent tid */
    tid?: string | null;
    /** WorkerEvent productType */
    productType?: string | null;
}

export declare type Level = "info" | "debug" | "warning" | "error";

export declare const enum LOG_TYPE {
    MISC = 0,
    INTERNAL_EVENT = 1,
    PUBLIC_EVENT = 2,
    WEB_EVENT = 3,
    INTERNAL_API = 4,
    WEB_API = 5,
    PUBLIC_API = 6
}

export declare enum LOG_UPLOAD_STATE {
    FREE = "free",
    UPLOADING = "uploading"
}

export declare interface LogData {
    timestamp: number;
    level: Level;
    type: LOG_TYPE;
    module: string;
    instanceId?: string | number;
}

export declare interface LogErrorDetail {
    status: number;
    message: string;
    data?: any;
    headers?: any;
    networkState?: NETWORK_STATE;
    errorRange: number[];
}

export declare const logger: AgoraLogger;

export declare type LogInstance = any | [string?, (string | number)?];

declare class LogReportBus extends EventEmitter {
    private currentUploadLogID;
    reportLogUploadError(errorDetail: LogErrorDetail): void;
}

export declare const logReportBus: LogReportBus;

declare type Long_2 = any;

export declare interface MiscLog extends LogData {
    type: LOG_TYPE.MISC;
    msg: string;
}

export declare type MiscUploadData = [...BaseUploadData, [string]];

declare type newReportMessages = IWebSocketQuit | IJoinWebProxyAP | IWorkerEvent | IAPWorkerEvent;

declare interface NotContainSuccessPropertyEventReportBase {
    sid: string;
    lts: number;
    cname: string | null;
    uid: number | null;
    peer?: number | null;
    cid: number | null;
    elapse: number | null;
    extend?: string | null;
    vid?: number;
    apiName?: string;
}

export declare interface PublicAPILog extends LogData {
    type: LOG_TYPE.PUBLIC_API;
    api: string;
    args?: string;
    result?: string;
}

export declare interface PublicEventLog extends LogData {
    type: LOG_TYPE.PUBLIC_EVENT;
    event: string;
    msg?: string;
}

export declare const report: AgoraRTCEventReport;

export declare interface UploadLog {
    payload_str: string;
    log_level: number;
    log_item_id: number;
}

export declare interface WebAPILog extends LogData {
    type: LOG_TYPE.WEB_API;
    api: string;
    args?: string;
    result?: string;
}

export declare interface WebEventLog extends LogData {
    type: LOG_TYPE.WEB_EVENT;
    event: string;
    msg?: string;
}

export { }
